extends Node2D

const UNIT := 32
const OFFSET := UNIT / 2.0
const SCREEN_WIDTH = 1152
const SCREEN_HEIGHT = 648
const WIDTH := SCREEN_WIDTH / UNIT - 1
const HEIGHT := SCREEN_HEIGHT / UNIT - 1
const DEFAULT_SPEED := 150.0

@onready var snake_body: Node2D = $"."
@onready var snake_head: Sprite2D = $Head

enum DIRECTION {
	UP,
	DOWN,
	LEFT,
	RIGHT,
}

var direction := DIRECTION.RIGHT
var previous_position: Vector2
var body_parts := Array()
var elapse := 0.0
var speed := DEFAULT_SPEED
var active := false

signal snake_grow(body_parts: Array)
signal snake_terminate

func _physics_process(delta: float) -> void:
	if not active:
		return
	elapse += delta * speed
	var step = floor(elapse / UNIT)
	if step == 0:
		return
	elapse = (elapse as int) % UNIT
	
	move()

func _input(_event) -> void:
	if direction == DIRECTION.LEFT or direction == DIRECTION.RIGHT:
		if Input.is_action_just_pressed("ui_up") and previous_position != snake_head.position:
			direction = DIRECTION.UP
			previous_position = snake_head.position
		elif Input.is_action_just_pressed("ui_down") and previous_position != snake_head.position:
			direction = DIRECTION.DOWN
			previous_position = snake_head.position
	if direction == DIRECTION.UP or direction == DIRECTION.DOWN:
		if Input.is_action_just_pressed("ui_left") and previous_position != snake_head.position:
			direction = DIRECTION.LEFT
			previous_position = snake_head.position
		elif Input.is_action_just_pressed("ui_right") and previous_position != snake_head.position:
			direction = DIRECTION.RIGHT
			previous_position = snake_head.position

func grow() -> void:
	var tail := Sprite2D.new()
	tail.texture = snake_head.texture
	tail.position = snake_head.position
	tail.scale = Vector2(0.9, 0.9)
	
	body_parts.append(tail)
	snake_body.add_child(tail)
	speed *= 1.05

	snake_grow.emit(body_parts)

func move() -> void:
	var current_head_position = Vector2(snake_head.position)
	if direction == DIRECTION.LEFT:
		snake_head.position.x -= UNIT
	elif direction == DIRECTION.RIGHT:
		snake_head.position.x += UNIT
	elif direction == DIRECTION.UP:
		snake_head.position.y -= UNIT
	else:
		snake_head.position.y += UNIT

	# cross screen
	if snake_head.position.x < 0:
		snake_head.position.x = WIDTH * UNIT + OFFSET
	elif snake_head.position.x > SCREEN_WIDTH:
		snake_head.position.x = OFFSET
	elif snake_head.position.y < 0:
		snake_head.position.y = HEIGHT * UNIT + OFFSET
	elif snake_head.position.y > SCREEN_HEIGHT:
		snake_head.position.y = OFFSET

	if is_collided(snake_head.position):
		snake_head.position = current_head_position
		snake_terminate.emit()
		return

	if not body_parts.is_empty():
		var last = body_parts.pop_back()
		last.position = current_head_position
		body_parts.insert(0, last)

func is_collided(new_position: Vector2):
	for body_part in body_parts:
		if body_part.position.x == new_position.x and body_part.position.y == new_position.y:
			return true
	return false

func restart():
	for body_part in body_parts:
		body_part.queue_free()
	body_parts = []

	var x = (floor(randf_range(0, WIDTH)) * UNIT + OFFSET) as int
	var y = (floor(randf_range(0, HEIGHT)) * UNIT + OFFSET) as int
	snake_head.position = Vector2(x, y)

	grow()

	speed = DEFAULT_SPEED
	active = true
	
