extends Node2D

# tetrominoes
var i_0 := [Vector2i(0, 1), Vector2i(1, 1), Vector2i(2, 1), Vector2i(3, 1)]
var i_90 := [Vector2i(2, 0), Vector2i(2, 1), Vector2i(2, 2), Vector2i(2, 3)]
var i_180 := [Vector2i(0, 2), Vector2i(1, 2), Vector2i(2, 2), Vector2i(3, 2)]
var i_270 := [Vector2i(1, 0), Vector2i(1, 1), Vector2i(1, 2), Vector2i(1, 3)]
var t_i := [i_0, i_90, i_180, i_270]

var t_0 := [Vector2i(1, 0), Vector2i(0, 1), Vector2i(1, 1), Vector2i(2, 1)]
var t_90 := [Vector2i(1, 0), Vector2i(1, 1), Vector2i(2, 1), Vector2i(1, 2)]
var t_180 := [Vector2i(0, 1), Vector2i(1, 1), Vector2i(2, 1), Vector2i(1, 2)]
var t_270 := [Vector2i(1, 0), Vector2i(0, 1), Vector2i(1, 1), Vector2i(1, 2)]
var t_t := [t_0, t_90, t_180, t_270]

var o_0 := [Vector2i(0, 0), Vector2i(1, 0), Vector2i(0, 1), Vector2i(1, 1)]
var o_90 := [Vector2i(0, 0), Vector2i(1, 0), Vector2i(0, 1), Vector2i(1, 1)]
var o_180 := [Vector2i(0, 0), Vector2i(1, 0), Vector2i(0, 1), Vector2i(1, 1)]
var o_270 := [Vector2i(0, 0), Vector2i(1, 0), Vector2i(0, 1), Vector2i(1, 1)]
var t_o := [o_0, o_90, o_180, o_270]

var z_0 := [Vector2i(0, 0), Vector2i(1, 0), Vector2i(1, 1), Vector2i(2, 1)]
var z_90 := [Vector2i(2, 0), Vector2i(1, 1), Vector2i(2, 1), Vector2i(1, 2)]
var z_180 := [Vector2i(0, 1), Vector2i(1, 1), Vector2i(1, 2), Vector2i(2, 2)]
var z_270 := [Vector2i(1, 0), Vector2i(0, 1), Vector2i(1, 1), Vector2i(0, 2)]
var t_z := [z_0, z_90, z_180, z_270]

var s_0 := [Vector2i(1, 0), Vector2i(2, 0), Vector2i(0, 1), Vector2i(1, 1)]
var s_90 := [Vector2i(1, 0), Vector2i(1, 1), Vector2i(2, 1), Vector2i(2, 2)]
var s_180 := [Vector2i(1, 1), Vector2i(2, 1), Vector2i(0, 2), Vector2i(1, 2)]
var s_270 := [Vector2i(0, 0), Vector2i(0, 1), Vector2i(1, 1), Vector2i(1, 2)]
var t_s := [s_0, s_90, s_180, s_270]

var l_0 := [Vector2i(2, 0), Vector2i(0, 1), Vector2i(1, 1), Vector2i(2, 1)]
var l_90 := [Vector2i(1, 0), Vector2i(1, 1), Vector2i(1, 2), Vector2i(2, 2)]
var l_180 := [Vector2i(0, 1), Vector2i(1, 1), Vector2i(2, 1), Vector2i(0, 2)]
var l_270 := [Vector2i(0, 0), Vector2i(1, 0), Vector2i(1, 1), Vector2i(1, 2)]
var t_l := [l_0, l_90, l_180, l_270]

var j_0 := [Vector2i(0, 0), Vector2i(0, 1), Vector2i(1, 1), Vector2i(2, 1)]
var j_90 := [Vector2i(1, 0), Vector2i(2, 0), Vector2i(1, 1), Vector2i(1, 2)]
var j_180 := [Vector2i(0, 1), Vector2i(1, 1), Vector2i(2, 1), Vector2i(2, 2)]
var j_270 := [Vector2i(1, 0), Vector2i(1, 1), Vector2i(0, 2), Vector2i(1, 2)]
var t_j := [j_0, j_90, j_180, j_270]

var shapes := []
var shapes_full := [t_i, t_t, t_o, t_z, t_s, t_l, t_j]

# grid variables
const COLS : int = 10
const ROWS : int = 20
const START_POSITION := Vector2i(5, 1)
const STEP : int = 50
const ACCEL : float = 0.25
const REWARD : int = 100

@onready var tile_map_layer_land : TileMapLayer = $TileMapLayerLand
@onready var tile_map_layer_active : TileMapLayer = $TileMapLayerActive
@onready var hud : CanvasLayer = $Hud

var current_position : Vector2i
var game_running : bool
var score : int
var speed : float
var steps_left : float
var steps_right : float
var steps_down : float

var piece_type : Array
var next_piece_type : Array
var piece_atlas : Vector2i
var next_piece_atlas : Vector2i
var rotation_index : int = 0
var active_piece : Array

func _ready():
	new_game()
	await Engine.get_process_frames()
	hud.get_node("Button").pressed.connect(new_game)

func _physics_process(_delta: float) -> void:
	if not game_running:
		return

	if Input.is_action_just_pressed("ui_up"):
		rotate_piece()

	if Input.is_action_pressed("ui_left"):
		steps_left += 10
	elif Input.is_action_pressed("ui_right"):
		steps_right += 10
	elif Input.is_action_pressed("ui_down"):
		steps_down += 10

	steps_down += speed

	if steps_left > STEP:
		move_piece(Vector2i.LEFT)
		steps_left = 0
	elif steps_right > STEP:
		move_piece(Vector2i.RIGHT)
		steps_right = 0
	elif steps_down > STEP:
		move_piece(Vector2i.DOWN)
		steps_down = 0

func new_game():
	# reset variables
	score = 0
	speed = 1.0
	game_running = true
	hud.get_node("GameOverLabel").hide()
	# clear everything
	clear_piece()
	clear_board()
	clear_panel()
	piece_type = pick_piece()
	piece_atlas = Vector2i(shapes_full.find(piece_type), 0)
	next_piece_type = pick_piece()
	next_piece_atlas = Vector2i(shapes_full.find(next_piece_type), 0)
	create_piece()

func clear_panel():
	for i in range(14, 19):
		for j in range(5, 9):
			tile_map_layer_active.erase_cell(Vector2i(i, j))

func pick_piece():
	if shapes.is_empty():
		shapes = shapes_full.duplicate()
		shapes.shuffle()

	return shapes.pop_front()

func create_piece():
	# reset variables
	current_position = START_POSITION
	active_piece = piece_type[rotation_index]

	draw_piece(active_piece, current_position, piece_atlas)
	# show next piece
	draw_piece(next_piece_type[0], Vector2i(15, 6), next_piece_atlas)

func rotate_piece():
	if can_rotate():
		clear_piece()
		rotation_index = (rotation_index + 1) % 4
		active_piece = piece_type[rotation_index]
		draw_piece(active_piece, current_position, piece_atlas)

func can_rotate():
	var temp_rotation_index = (rotation_index + 1) % 4
	for i in piece_type[temp_rotation_index]:
		if not is_free(i + current_position):
			return false
	return true

func is_free(pos):
	return tile_map_layer_land.get_cell_source_id(pos) == -1

func move_piece(dir):
	if can_move(dir):
		clear_piece()
		current_position += dir
		draw_piece(active_piece, current_position, piece_atlas)
	else:
		if dir == Vector2i.DOWN:
			land_piece()
			check_rows()
			piece_type = next_piece_type
			piece_atlas = next_piece_atlas
			next_piece_type = pick_piece()
			next_piece_atlas = Vector2i(shapes_full.find(next_piece_type), 0)
			clear_panel()
			create_piece()
			check_game_over()

func can_move(dir):
	# check if there is space to move
	for i in active_piece:
		if not is_free(i + current_position + dir):
			return false
	return true

func land_piece():
	# remove each segment from the active layer and move to board layer
	for i in active_piece:
		tile_map_layer_active.erase_cell(current_position + i)
		tile_map_layer_land.set_cell(current_position + i, 0, piece_atlas)

func check_rows():
	var row : int = ROWS
	while row > 0:
		var count = 0
		for i in range(COLS):
			if not is_free(Vector2i(i + 1, row)):
				count += 1
		# if row is full then erase it
		if count == COLS:
			shift_rows(row)
			score += REWARD
			hud.get_node("ScoreLabel").text = "SCORE: " + str(score)
			speed += ACCEL
		else:
			row -= 1

func shift_rows(row):
	var atlas
	for i in range(row, 1, -1):
		for j in range(COLS):
			atlas = tile_map_layer_land.get_cell_atlas_coords(Vector2i(j + 1, i - 1))
			if atlas == Vector2i(-1, -1):
				tile_map_layer_land.erase_cell(Vector2i(j + 1, i))
			else:
				tile_map_layer_land.set_cell(Vector2i(j + 1, i), 0, atlas)

func clear_piece():
	for i in active_piece:
		tile_map_layer_active.erase_cell(current_position + i)

func draw_piece(piece, pos, atlas):
	for i in piece:
		tile_map_layer_active.set_cell(pos + i, 0, atlas)

func clear_board():
	for i in range(ROWS):
		for j in range(COLS):
			tile_map_layer_land.erase_cell(Vector2i(j + 1, i + 1))

func check_game_over():
	for i in active_piece:
		if not is_free(i + current_position):
			land_piece()
			hud.get_node("GameOverLabel").show()
			game_running = false
